
;;; AUTOMATES CELLULAIRES LMENTAIRES (UNI-DIMESIONNELS)

;;   Cf. [DELORME, M., & MAZOYER, J., 2003 : "La riche zoologie des automates
;; cellulaires", Pour la science : 314 : 40-45].

;;; Charger la librairie QuickDraw.

(defun cat-pathnames (depart &rest frags)
  (labels ((interne (frgs ldr)
            (if (endp (cdr frgs))
              (merge-pathnames (car frgs) ldr)
            (interne (cdr frgs)
                     (merge-pathnames 
                      (make-pathname
                       :directory
                       (namestring (merge-pathnames (car frgs) ldr)))
                      "*")))))
  (interne frags depart)))
(load (cat-pathnames (mac-default-directory) "Library" "QuickDraw"))
(format t "~&;Librarie QuickDraw charge...~%")

;; Dfinitions de variables lies  la reprsentation graphique
(defparameter points 5)  ;taille des caractres
(defparameter Ox 0)      ;origine axe des x (cellules)
(defparameter Oy 4)      ;origine axe des y (etats)
;(defparameter car "g")   ;ceci inutile, juste pour voir... (cf. ci-dessous)
(defparameter Wg "g")    ;carr noir en police Webdings
(defparameter Wc "c")    ;carr blanc en police Webdings

;;; Fonction principale

;;   Ralise un automate cellulaire dans un tableau bidimensionnel.
;; <RegleDec> := rgle de l'automate, exprime en dcimal. Ex . : 184 = rgle
;;               de la "circulation automobile" := (1 0 1 1 1 0 0 0) := prochain
;;               tat du bit central des 8 triplets de birs possible, de 111 
;;               000, selon la notation traditionnelle [op.cit. : 42]. Entier
;;               dans [0,255].
;; Les mots-clef suivants sont possibles :
;; :EtatInitial <li> := permet de spcifier l'tat initial des cellules, sous
;;                      forme d'une liste <li> de <cellules> 0 et 1 [par dfaut
;;                      nil].
;; :cellules <nbr>   := nombre de cellules, entier dans [5,226]. Correspond  la
;;                      dimension horizontale de la fentre d'affichage [par
;;                      dfaut 226].
;; :etats <nbr>      := nombre d'tats affichables verticalement dans la fentre
;;                      d'affichage [par dfaut 140].
;; :etapes<nbr>      := nombre d'tapes (tats successifs)  calculer pour
;;                      l'automate [par dfaut 999999].
;;   Faire Abort pour interrompre, mais risque de plantage... Plus prudent de
;; limiter le nombre d'tapes (ex.: 280).
(defun AutoCell (RegleDec
                 &key EtatInitial (cellules 226) (etats 140) (etapes 999999)
                 &aux tmp titre)
  ;Dtection de valeurs inappropries
  (when (or (not (integerp RegleDec)) (< RegleDec 0) (> RegleDec 255))
    (error "AutoCell : regle doit etre un entier dans [0,255]."))
  (when (or (not (integerp cellules)) (< cellules 5) (> cellules 226))
    (error "AutoCell : nombre de cellules doit etre un entier dans [5,226]."))
  (when (or (not (integerp etats)) (< etats 5) (> etats 140))
    (error "AutoCell : nombre d'etats doit etre un entier dans [5,140]."))
  (when (and EtatInitial (/= cellules (list-length EtatInitial)))
    (error "AutoCell : l'etat initial n'est pas de ~D cellules." cellules))
  ;Cration d'un <EtatInitial> s'il n'a pas t donn
  (setq titre "donne.")
  (when (not EtatInitial) (setq EtatInitial (AleaEtatInit cellules))
                          (setq titre "aleatoire."))
  ;TabRegle = reprsentation binaire de la rgle <RegleDec> donne, selon
  ;           l'ordre normal des triplets de bits de 111  000 (rgles locales).
  ;           TabRegle[N] contient l'tat suivant de la cellule centrale du
  ;           triplet de bits dont la valeur dcimale est N.
  ;GdTab = tableau de <cellules> * <etats>. Il contiendra des 0 et 1
  ;        reprsentant les valeurs des cellules  chaque tat de l'automate.
  ;        L'affichage dans la fentre qui sera dfinie correspondra  GdTab.
  ;*view* = pointeur sur la fentre d'affichage des tats de l'automate.
  ;etpA, etpF = compteurs d'tapes (Actuelle et Future).
  (let ((TabRegle (make-array
                   8
                   :initial-contents
                    (reverse
                     (append (make-list (- 8 (list-length
                                              (setq tmp (dec>bin RegleDec))))
                                        :initial-element 0)
                             tmp))))
        (GdTab (make-array (list cellules etats)))
        (*view* (cree-fenetre cellules etats RegleDec titre))
        etpA etpF)
    ;Transcription de la liste <EtatInitial> dans les cellules de l'etat 0 du
    ;tableau GdTab
    (setq tmp -1)
    (dolist (cell EtatInitial)
      (setf (aref GdTab (incf tmp) 0) cell))
;(dbg EtatInitial)
;(dbg cellules)
;(dbg etats)
;(dbg etapes)
;(dbg TabRegle)
;(dotimes (n 8 'ok)
;  (prins "TabRegle[" n "] = " (aref TabRegle n)))
;(dbg GdTab)
;(dbg *view*)
    ;Boucle des <etapes> tapes successives
    (dotimes (etp etapes)
      (setq etpA (mod etp etats))         ;tape Actuelle
      (setq etpF (mod (1+ etpA) etats))   ;tape Future
      ;Affichage de l'tape Actuelle
      (move-to *view* Ox (+ Oy (* etpA points)))
      (dotimes (c cellules)
        (format *view* (if (zerop (aref GdTab c etpA)) Wc Wg)))
      ;Cellules : prparation de l'tape Future
      ;Deux cas particuliers pour la premire et la dernire des cellules : leur
      ;triplet de rgle locale tient compte du bit situ  l'autre extrmit de
      ;la range de cellules...
      (setf (aref GdTab 0              etpF)
            (aref TabRegle (bin>dec (list (aref GdTab (- cellules 1) etpA)
                                          (aref GdTab 0              etpA)
                                          (aref GdTab 1              etpA)))))
      (setf (aref GdTab (- cellules 1) etpF)
            (aref TabRegle (bin>dec (list (aref GdTab (- cellules 2) etpA)
                                          (aref GdTab (- cellules 1) etpA)
                                          (aref GdTab 0              etpA)))))
      ;Boucle des <cellules>-2 autrs cellules
      (dotimes (c (- cellules 2))
        (setf (aref GdTab (setq tmp (+ c 1)) etpF)
              (aref TabRegle (bin>dec (list (aref GdTab c       etpA)
                                            (aref GdTab tmp     etpA)
                                            (aref GdTab (+ c 2) etpA))))))      
)))

;;; Utilitaires

;;   Pour crer un <EtatInitial> alatoire : une liste de longueur <nbr>
;; compose de 0 et 1 quiprobables.
(defun AleaEtatInit (nbr &aux rez)
  (dotimes (n nbr rez)
    (push (random 2) rez)))

;;   Conversion binaire vers dcimal. Cf. Chap. 12.
;; Exemple : (bin>dec '(1 1 0 1 0 1)) => 53.
(defun bin>dec (li)
  (labels ((auXil (li res)                           ;dfinition de auXil
             (if li
               (auXil (cdr li) (+  (* res 2)         ;calcul et rcursion...
                                   (car li)))
             res)))                                  ;cas terminal
    (auXil li 0)))                                   ;corps du labels :

;;   Conversion dcimal vers binaire.
;; Exemple : (bin>dec (dec>bin 342476774289)) => 342476774289
(defun dec>bin (nbr &aux (tmp 0))
  (let* ((lng (if (<= nbr 0) 0 (floor (log nbr 2))))
         (rez (make-list (1+ lng) :initial-element 0)))
    (labels ((interne (n)
               (if (<= n 0) (reverse rez)
               (progn (setf (nth (setq tmp (floor (log n 2))) rez) 1)
                      (interne (- n (expt 2 tmp)))))))
      (if (and (integerp nbr) (> nbr 0)) (interne nbr) rez))))

;; Recherches, essais... rfrences au manuel.
; *font-list* => ... "Webdings"... p. 74
; (set-view-font <simple-view> <font-specs>) p. 79
; font-specs ("Webdings" 9 :srcCopy (:color <x>)) p. 75
; (make-color <r> <g> <b>)  dans 0,65535 p. 251
; ex : (make-color 764 6543 54365) => 203220
; font-specs ("Webdings" 9 :srcCopy (:color 203220)) p. 75
; (font-codes '("Webdings" 9 :srcCopy (:color 203220))) = OK ?
; (view-font *fen*)

;;   Cration d'une fentre pour la reprsentation graphique.
;; Horizontalement : <cells> cellules ; verticalement : <etats> tats successifs
;; de l'automate. La succession des tats (tapes) est affiche verticalement de
;; haut en bas. Au-del du nombre d'tats affichables dans la fentre,
;; l'affichage boucle en haut de la fentre.
;;   <regle> et <titre> feront partie du titre de la fentre.
;;   L'utilitaire int>string est dfini dans +clinit.cl ; cf. aussi Chap. 14.
;; Exemple : (int>string 40781) => "40781"
(defun cree-fenetre (cells etats regle titre &aux *fen*)
  (setq cells (* points cells))
  (setq etats (* points etats))
  (setq regle (int>string regle))
  (setq *fen*
    (make-instance 'window ;cf. p 154 classe window
                   :view-size (make-point cells etats)
                   :window-title
                    (concatenate 'string "Automate cellulaire, regle "
                                         regle
                                         ", etat initial "
                                         titre)
                   :view-font '("Webdings" 5 :srcCopy (:color-index 211))
                   :color-p t
                   :window-type :document
                   :close-box-p t))
  (set-fore-color *fen* 5062365)
  (set-back-color *fen* 16318223)
  (zigonne *fen*)                                          ;pour viter la perte
  (fill-rect *fen* *light-gray-pattern* 0 0 cells etats)   ;griser la fentre
;Ceci inutile, juste pour voir fentre lors des premiers tests...
;  (dotimes (n (/ etats points) 'ok) ;etats
;    (move-to *fen* Ox (+ Oy (* n points)))
;    (dotimes (n (/ cells points) 'ok) ;cellules
;      (format *fen* (if (string-equal car "g") (setq car "c") (setq car "g")))))
;
  *fen*
)

;;   Mouvement inutiles  effectuer aprs la cration de la fentre, vitant la
;; perte des quelques premiers affichages. Empirique et inexplicable !!!
(defun zigonne (fenetre)
  (dotimes (n 100001 t)                   ;100001 = empirique & inexplicable !!!
    (move fenetre 100 100)))

;;; Exemples 140 tapes (1 fentre = 226 x 140 par dfaut)

;; Possible en continu -- sans :etapes <n>
;; Dans ce cas : Arrt <= Abort (danger plantage... !)

;;; Classe de Wolfram 1 : uniformes ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Tout "blanc"
; (AutoCell   0 :etapes 140) blanc immdiat, videmment 
; (AutoCell   8 :etapes 140) (AutoCell  32 :etapes 140)
; (AutoCell  40 :etapes 140) (AutoCell  64 :etapes 140)
; (AutoCell  96 :etapes 140)
; (AutoCell 128 :etapes 140) (AutoCell 136 :etapes 140)
; (AutoCell 160 :etapes 140) (AutoCell 168 :etapes 140)
; (AutoCell 192 :etapes 140) (AutoCell 224 :etapes 140)
;Tout "noir"
; (AutoCell 234 :etapes 140) (AutoCell 235 :etapes 140)
; (AutoCell 238 :etapes 140) (AutoCell 239 :etapes 140)
; (AutoCell 248 :etapes 140) (AutoCell 249 :etapes 140)
; (AutoCell 250 :etapes 140) (AutoCell 251 :etapes 140)
; (AutoCell 252 :etapes 140) (AutoCell 253 :etapes 140)
; (AutoCell 254 :etapes 140)
; (AutoCell 255 :etapes 140) noir immdiat, videmment

;;; Classe de Wolfram 2 : priodiques (les plus nombreux) ;;;;;;;;;;;;;;;;;;;;;;

; (AutoCell   1 :etapes 140) avec oscillations internes
;Alterne blanc/noir si tat initial tout noir/blanc
; (AutoCell   1 :etapes 200 :cellules 100 :etats 100 :EtatInitial (make-list 100 :initial-element 1))
; (AutoCell   1 :etapes 200 :cellules 100 :etats 100 :EtatInitial (make-list 100 :initial-element 0))
;Cf. [KICINGER, R. & ARCISZEWSKI, T., 2008 : "La gntique au service des
;architectes", Pour la science : 363 : 32-38], fig. 5 droite, p. 36.
;N.B. : toutes les reconstitutions des figures de cet article sont faites ici de
;haut en bas, tandis que l'article les prsente de bas en haut.
; (Autocell   1 :cellules 5 :etats 30  :etapes 30 :EtatInitial '(0 0 1 0 0))
;Autres...
; (AutoCell   2 :etapes 140) (AutoCell   3 :etapes 140)
; (AutoCell   4 :etapes 140) (AutoCell   5 :etapes 140)
; (AutoCell   6 :etapes 140) (AutoCell   7 :etapes 140)
; (AutoCell   9 :etapes 140) (AutoCell  10 :etapes 140)
; (AutoCell  11 :etapes 140) "curly diags"
; (AutoCell  12 :etapes 140)
; (AutoCell  13 :etapes 140) (AutoCell  14 :etapes 140)
; (AutoCell  15 :etapes 140) (AutoCell  16 :etapes 140)
; (AutoCell  17 :etapes 140) (AutoCell  19 :etapes 140)
; (AutoCell  20 :etapes 140) (AutoCell  21 :etapes 140)
; (AutoCell  23 :etapes 140)
;Cf. [KICINGER, R. & ARCISZEWSKI, T.], fig. 5 centre, p. 36 :
; (Autocell  23 :cellules 5 :etats 30  :etapes 30 :EtatInitial '(0 0 1 0 0))
; (AutoCell  24 :etapes 140) (AutoCell  25 :etapes 140)
; (AutoCell  26 :etapes 140) priodique, mais inclut aussi des triangles !
; (AutoCell  27 :etapes 140)
; (AutoCell  28 :etapes 140) (AutoCell  29 :etapes 140)
; (AutoCell  31 :etapes 140) (AutoCell  33 :etapes 140)
; (AutoCell  34 :etapes 140) (AutoCell  35 :etapes 140)
; (AutoCell  36 :etapes 140) (AutoCell  37 :etapes 140)
; (AutoCell  38 :etapes 140) (AutoCell  39 :etapes 140)
; (AutoCell  41 :etapes 140) ;stabilisation lente
; (AutoCell  42 :etapes 140) (AutoCell  43 :etapes 140)
; (AutoCell  44 :etapes 140) (AutoCell  46 :etapes 140)
; (AutoCell  47 :etapes 140) ;"curly diags"
; (AutoCell  48 :etapes 140) (AutoCell  49 :etapes 140)
; (AutoCell  50 :etapes 140)
;Cf. [KICINGER, R. & ARCISZEWSKI, T.], fig. 7 extrme droite, p. 38 :
; (Autocell 50 :cellules 5 :etats 30  :etapes 30 :EtatInitial '(0 0 1 0 0))
; (AutoCell  51 :etapes 140)
; (AutoCell  52 :etapes 140) (AutoCell  53 :etapes 140)
; (AutoCell  55 :etapes 140) (AutoCell  56 :etapes 140)
; (AutoCell  57 :etapes 140) (AutoCell  58 :etapes 140)
; (AutoCell  59 :etapes 140) (AutoCell  61 :etapes 140)
; (AutoCell  62 :etapes 140) stabilisation lente, commence comme 54 (pluie)
; (AutoCell  63 :etapes 140) (AutoCell  65 :etapes 140)
; (AutoCell  66 :etapes 140) (AutoCell  67 :etapes 140)
; (AutoCell  68 :etapes 140) (AutoCell  69 :etapes 140)
; (AutoCell  70 :etapes 140) (AutoCell  71 :etapes 140)
; (AutoCell  72 :etapes 140)
; (AutoCell  73 :etapes 140) priodique, mais colonnes varies !
; (AutoCell  74 :etapes 140) (AutoCell  76 :etapes 140)
; (AutoCell  77 :etapes 140) (AutoCell  78 :etapes 140)
; (AutoCell  79 :etapes 140) (AutoCell  80 :etapes 140)
; (AutoCell  81 :etapes 140) "curly diags"
; (AutoCell  82 :etapes 140) priodique, mais oscille aussi en triangles !
; (AutoCell  83 :etapes 140)
; (AutoCell  84 :etapes 140) stabilisation assez lente
; (AutoCell  85 :etapes 140) "curly diags"
; (AutoCell  87 :etapes 140)
; (AutoCell  88 :etapes 140)
; (AutoCell  91 :etapes 140) (AutoCell  92 :etapes 140)
; (AutoCell  93 :etapes 140) (AutoCell  94 :etapes 140) ;avec qqqs triangles
; (AutoCell  95 :etapes 140) (AutoCell  98 :etapes 140)
; (AutoCell  99 :etapes 140) (AutoCell 100 :etapes 140)
; (AutoCell 103 :etapes 140) (AutoCell 104 :etapes 140)
; (AutoCell 107 :etapes 140) stabilisation lente
; (AutoCell 108 :etapes 140)
; (AutoCell 109 :etapes 140) avec oscillations priodiques (autres ci-dessus)
; (AutoCell 111 :etapes 140) (AutoCell 112 :etapes 140)
; (AutoCell 113 :etapes 140) (AutoCell 114 :etapes 140)
; (AutoCell 115 :etapes 140) (AutoCell 116 :etapes 140)
; (AutoCell 117 :etapes 140)
; (AutoCell 118 :etapes 140) stabilisation lente, cf. 62 (et donc 54)
; (AutoCell 119 :etapes 140) (AutoCell 120 :etapes 140)
; (AutoCell 121 :etapes 140) stabilisation lente
; (AutoCell 123 :etapes 140) (AutoCell 125 :etapes 140)
; (AutoCell 127 :etapes 140) (AutoCell 130 :etapes 140)
; (AutoCell 131 :etapes 140) stabilisation lente
; (AutoCell 132 :etapes 140) (AutoCell 133 :etapes 140)
; (AutoCell 134 :etapes 140)
; (AutoCell 138 :etapes 140) (AutoCell 139 :etapes 140) ;diagonales
; (AutoCell 140 :etapes 140) (AutoCell 141 :etapes 140)
; (AutoCell 142 :etapes 140) (AutoCell 143 :etapes 140)
; (AutoCell 144 :etapes 140)
; (AutoCell 145 :etapes 140) stabilisation lente, cf. 62 (et donc 54)
; (AutoCell 148 :etapes 140) (AutoCell 152 :etapes 140)
; (AutoCell 154 :etapes 140) ;diagonales  motifs
; (AutoCell 155 :etapes 140) (AutoCell 156 :etapes 140)
; (AutoCell 157 :etapes 140) (AutoCell 158 :etapes 140)
; (AutoCell 159 :etapes 140) (AutoCell 162 :etapes 140)
; (AutoCell 163 :etapes 140) (AutoCell 164 :etapes 140)
; (AutoCell 166 :etapes 140) avec oscillations internes (cf. 82 etc. ci-dessus)
; (AutoCell 167 :etapes 140) avec oscillations internes (cf. 82 etc. ci-dessus)
; (AutoCell 170 :etapes 140) Exemple [op.cit. : 44], vers la gauche...
; (AutoCell 171 :etapes 140) (AutoCell 172 :etapes 140)
; (AutoCell 173 :etapes 140) avec oscillations internes (cf. etc. ci-dessus)
; (AutoCell 174 :etapes 140) (AutoCell 175 :etapes 140)
; (AutoCell 176 :etapes 140) (AutoCell 177 :etapes 140)
; (AutoCell 178 :etapes 140) avec oscillations internes (cf. etc. ci-dessus)
; (AutoCell 179 :etapes 140) idem
; (AutoCell 180 :etapes 140) idem
; (AutoCell 181 :etapes 140) idem
; (AutoCell 184 :etapes 140) Exemple de la circulation automobile [op.cit. : 40]
; (AutoCell 185 :etapes 140) (AutoCell 186 :etapes 140)
; (AutoCell 187 :etapes 140) (AutoCell 188 :etapes 140)
; (AutoCell 189 :etapes 140) (AutoCell 190 :etapes 140)
; (AutoCell 191 :etapes 140) (AutoCell 194 :etapes 140)
; (AutoCell 196 :etapes 140) (AutoCell 197 :etapes 140)
; (AutoCell 198 :etapes 140) avec oscillations internes (cf. etc. ci-dessus)
; (AutoCell 199 :etapes 140) 2 semblables de suite !
; (AutoCell 200 :etapes 140)
; (AutoCell 201 :etapes 140) avec oscillations internes (cf. etc. ci-dessus)
; (AutoCell 202 :etapes 140) (AutoCell 203 :etapes 140)
; (AutoCell 204 :etapes 140) Exemple [op.cit. : 44]
; (AutoCell 205 :etapes 140) (AutoCell 206 :etapes 140)
; (AutoCell 207 :etapes 140) (AutoCell 208 :etapes 140)
; (AutoCell 209 :etapes 140)
; (AutoCell 210 :etapes 140) avec oscillations internes (cf. etc. ci-dessus)
; (AutoCell 211 :etapes 140)
; (AutoCell 212 :etapes 140) ondulations amorties -> diagonales
; (AutoCell 213 :etapes 140) ondulations
; (AutoCell 214 :etapes 140) diagonales
; (AutoCell 215 :etapes 140) oscillations internes
; 216  223. : plusieurs semblables de suite, noirs avec vertivales blanches
; (AutoCell 216 :etapes 140) (AutoCell 217 :etapes 140)
; (AutoCell 218 :etapes 140) (AutoCell 219 :etapes 140)
; (AutoCell 220 :etapes 140) (AutoCell 221 :etapes 140)
; (AutoCell 222 :etapes 140) (AutoCell 223 :etapes 140)
;
; (AutoCell 226 :etapes 140) (AutoCell 227 :etapes 140)
; (AutoCell 228 :etapes 140) (AutoCell 229 :etapes 140)
; (AutoCell 230 :etapes 140) (AutoCell 231 :etapes 140)
; (AutoCell 232 :etapes 140) (AutoCell 233 :etapes 140)
; (AutoCell 236 :etapes 140)
; (AutoCell 237 :etapes 140) autre noir  vert. bl.
; (AutoCell 240 :etapes 140) Exemple [op.cit. : 44], vers la droite...
;   Cette rgle est dite"moins complexe" que la rgle 54, car contenue dans
;   cette dernire
; (AutoCell 241 :etapes 140)
; (AutoCell 242 :etapes 140) (AutoCell 243 :etapes 140)
; (AutoCell 244 :etapes 140) 
; (AutoCell 245 :etapes 140) (AutoCell 246 :etapes 140)
; (AutoCell 247 :etapes 140)

;;; Classe de Wolfram 3 : chaotiques "compliqus" ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; (AutoCell  75 :etapes 140) (AutoCell  86 :etapes 140)
; (AutoCell 110 :etapes 140)
; (AutoCell 126 :etapes 140)
; (AutoCell 150 :etapes 140) Exemple Fig. 3 [op.cit. : 42]
;Comparer deux squences de 140 tapes, avec tats initiaux diffrant d'une
;    seule cellule => rpercussions jusqu'aux points les plus loigns...
;    Cf. Fig. 3 [op.cit. : 42]
; (AutoCell 150 :EtatInitial '(0 1 0 0 1 1 1 0 0 1 0 0 0 1 0 1 1 0 0 1 1 0 1 0 1 0 0 1 0 1 0 0 0 1 0 1 1 0 0 1 0 1 0 1 0 1 1 0 0 0 0 1 1 0 0 1 0 0 0 0 0 1 0 1 1 1 1 0 0 1 0 0 1 0 0 0 1 0 0 0 0 0 1 1 1 1 0 0 1 1 0 1 1 1 0 0 1 0 1 1 0 1 1 0 0 1 0 0 1 1 0 0 0 1 1 1 1 0 0 0 0 1 0 0 1 0 0 0 0 1 1 1 0 0 0 0 1 0 1 0 0 1 1 1 1 1 0 0 1 0 0 1 1 1 1 1 0 1 1 0 0 1 0 0 1 0 0 1 0 0 1 0 0 1 0 1 1 0 0 0 0 0 0 0 1 0 1 1 1 1 1 0 0 0 0 1 0 0 1 1 0 0 1 1 1 0 0 0 1 0 1 1 1 0 1 1 1 0 1 1 0 0 0 0 1 1) :cellules 226 :etats 140 :etapes 140)
; (AutoCell 150 :EtatInitial '(0 1 0 0 1 1 1 0 0 1 0 0 0 1 0 1 1 0 0 1 1 0 1 0 1 0 0 1 0 1 0 0 0 1 0 1 1 0 0 1 0 1 0 1 0 1 1 0 0 0 0 1 1 0 0 1 0 0 0 0 0 1 0 1 1 1 1 0 0 1 0 0 1 0 0 0 1 0 0 0 0 0 1 1 1 1 0 0 1 1 0 1 1 1 0 0 1 0 1 1 0 1 1 0 0 0 0 0 1 1 0 0 0 1 1 1 1 0 0 0 0 1 0 0 1 0 0 0 0 1 1 1 0 0 0 0 1 0 1 0 0 1 1 1 1 1 0 0 1 0 0 1 1 1 1 1 0 1 1 0 0 1 0 0 1 0 0 1 0 0 1 0 0 1 0 1 1 0 0 0 0 0 0 0 1 0 1 1 1 1 1 0 0 0 0 1 0 0 1 1 0 0 1 1 1 0 0 0 1 0 1 1 1 0 1 1 1 0 1 1 0 0 0 0 1 1) :cellules 226 :etats 140 :etapes 140)
;    ->  pour voir la diffrence...  ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   ->   -> ici : ^
;    ... elle est sous le "re" de "cellulaire" dans le titre des fentres
; (AutoCell 225 :etapes 140)

;;; Classe de Wolfram 4 : complexes ("vritable complexit") [op.cit. : 45] ou
;;; " particules" triangulaires [op.cit. : 42] ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;Le 54 inclut les 170, 204 et 240.
; (AutoCell  54 :etapes 140) Exemple Fig. 1 [op.cit. : 41] et Fig. 5 [op.cit. : 44] :
;    gouttes d'eau sur une surface vitre.
;MAIS : tout blanc si tat initial tout noir ou tout blanc...
; (AutoCell  54 :etapes 400 :etats 100 :cellules 100 :EtatInitial (make-list 100 :initial-element 1))
; (AutoCell  54 :etapes 400 :etats 100 :cellules 100 :EtatInitial (make-list 100 :initial-element 0))
; (AutoCell 110 :etapes 140) Exemple Fig. 8 [op.cit. : 45] "Turing universel"
;Aussi rgle 110, dpart avec un seul noir au milieu cf. [POSTEL-VINAY, O.,
;2003 : "1. Les nouveaux dmiurges", La recherche : 360 : 35]
; (AutoCell 110 :etapes 900 :cellules 226 :EtatInitial (let ((dep (make-list 226 :initial-element 0))) (rplaca (nthcdr 225 dep) 1) dep))
;MAIS : tout blanc si tat initial tout noir ou tout blanc...
; (AutoCell 110 :etapes 100 :etats 100 :cellules 100 :EtatInitial (make-list 100 :initial-element 1))
; (AutoCell 110 :etapes 100 :etats 100 :cellules 100 :EtatInitial (make-list 100 :initial-element 0))
; (AutoCell 137 :etapes 140)
; (AutoCell 147 :etapes 140) ; semblable au 54, mais  gouttes noires

;;; Complexes, Classes 3 ou 4 tests empiriquement ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; (AutoCell  18 :etapes 140) (AutoCell  22 :etapes 140)
; (AutoCell  30 :etapes 140)
; (AutoCell  60 :etapes 140)
; (AutoCell  90 :etapes 140) (AutoCell 102 :etapes 140)
; (AutoCell 105 :etapes 140) ;assez original
;Cf. [KICINGER, R. & ARCISZEWSKI, T.], fig. 5 gauche, p. 36. Mais erreur dans
;l'article : il s'agit de la rgle 105, et non 150 : 
; (Autocell 105 :cellules 5 :etats 30  :etapes 30 :EtatInitial '(0 0 1 0 0))

; (AutoCell 106 :etapes 140) (AutoCell 122 :etapes 140)
; (AutoCell 124 :etapes 140) (AutoCell 129 :etapes 140)
; (AutoCell 135 :etapes 140) (AutoCell 146 :etapes 140)
; (AutoCell 149 :etapes 140) (AutoCell 151 :etapes 140)
; (AutoCell 153 :etapes 140) (AutoCell 161 :etapes 140)
; (AutoCell 165 :etapes 140) (AutoCell 169 :etapes 140)
; (AutoCell 182 :etapes 140) (AutoCell 183 :etapes 140) ;2 semblables de suite !
; (AutoCell 193 :etapes 140) (AutoCell 195 :etapes 140)

;;; Complexes, probablement classe 3 : cf. 75
; (AutoCell  45 :etapes 140)
; (AutoCell  89 :etapes 140) (AutoCell  97 :etapes 140)
; (AutoCell 101 :etapes 140)

;;; Complexes, probablement classe 3

; (AutoCell 120 :etapes 140)
