;;; Fichier  charger. Ouvre la fentre
;;; fen de COLONS x LIGNES cellules

;;; *GAME OF LIFE* (Jeu de la vie) de John Horton Conway (1970)

;;   Cf. http://en.wikipedia.org/wiki/Conway's_Game_of_Life et [DELAHAYE,
;; Jean-Paul, 2006 : "Concevoir l'univers comme un ordinateur ?", Pour la
;; science : 349 : 92-93].

;;   Dfinitions de variables globales lies  la reprsentation graphique

(defparameter COLONS 91) ;nombre de colonnes (largeur) tableau lebensraum
(defparameter LIGNES 57) ;nombre de lignes (hauteur)
;Cration du tableau 3 dimensions lebensraum :
;2 "tats" x COLONS colonnes x LIGNES lignes,
;(2 tats utiliss alternativement pour stocker les gnrations successives).
;La variable globale *etat* prendra alternativement les valeurs 0 et 1
;pour dsigner la premire dimension du tableau lebensraum. Les tats initiaux
;seront toujours placs dans l'tat 0. Plusieurs fonctions ci-dessous mettent
;*etat*  0. Seule la fonction generation fait alterner *etat* entre 0 et 1.
;*etat* sera toujours le nouvel tat affich par la fonction affiche.
(defparameter lebensraum (make-array (list 2 COLONS LIGNES)))
(defparameter *etat* 0)
;Variables graphiques
(defparameter *fen* nil) ;fentre d'affichage, dfinie ci-dessous fin de fichier
(defparameter points 5)  ;taille pixels des caractres reprsentant les cellules
(defparameter Ox 0)      ;origine axe des x (colonnes)
(defparameter Oy 4)      ;origine axe des y (lignes)
(defparameter Wg "g")    ;carr noir en police Webdings = cellule occupe = 1
(defparameter Wc "c")    ;carr blanc en police Webdings = cellule vide = 0
(defparameter *rall* 1000000) ;boucle -> ralentissement de l'affichage

;;   Mise  zro du tableau lebensraum. Amlioration 07/2008 : ajout affichage
;; de toutes les cellules en tat 0 par la fonction affiche-vide.

(defun vide-lebensraum ()
  (dotimes (e 2 'vide)
    (dotimes (l LIGNES)
      (dotimes (c COLONS)
        (setf (aref lebensraum e c l) 0))))
  (affiche-vide))

;;   Fonction dcomptant le nombre de cellules occupes parmi les huit voisines
;; de la cellule [<col>;<lin>] de l'un des deux tats <e> (0 ou 1). Dans
;; l'ordre suivant : ---col--> |
;;                   01 02 03  |
;;                   04    05 lin
;;                   06 07 08  |
;;                             V
;; N.B. : l'origine des coordonnes correspondra au coin SUPRIEUR GAUCHE de la
;; fentre d'affichage. Cellule vide = 0 ; occupe = 1.

(defun nbr-voisines (e col lin)
  (let ((c-1 (mod (1- col) COLONS))
        (c+1 (mod (1+ col) COLONS))
        (l-1 (mod (1- lin) LIGNES))
        (l+1 (mod (1+ lin) LIGNES)))
    (+ (aref lebensraum e c-1 l-1)         ;01
       (aref lebensraum e col l-1)         ;02
       (aref lebensraum e c+1 l-1)         ;03
       (aref lebensraum e c-1 lin)         ;04
       (aref lebensraum e c+1 lin)         ;05
       (aref lebensraum e c-1 l+1)         ;06
       (aref lebensraum e col l+1)         ;07
       (aref lebensraum e c+1 l+1))))      ;08

;;   Fonction dcidant du "destin" d'une cellule [<col>;<lin>] de l'un des deux
;; tats <e> (0 ou 1)  la gnration suivante. Rend 0 ou 1.

(defun destin (e col lin)
  (let ((nbr (nbr-voisines e col lin)))
    (setf (aref lebensraum (if (= e 0) 1 0) col lin)
          (if (= 0 (aref lebensraum e col lin)) (if (= 3 nbr) 1 0)
              (if (or (= 2 nbr) (= 3 nbr)) 1 0)))))

;;   Fait voluer l'tat *etat* d'une gnration, celle-ci tant stocke dans
;; l'autre tat du tableau lebensraum.
;; N.B. : FAIT AUSSI ALTERNER LA VALEUR DE *etat*.

(defun generation (&aux e-nouveau)
  (setq e-nouveau (if (= *etat* 0) 1 0))
  (dotimes (c COLONS)
    (dotimes (l LIGNES)
      (setf (aref lebensraum e-nouveau c l) (destin *etat* c l))))
  (setq *etat* e-nouveau))

;;   Lance une succession de <n> gnrations  partir de l'tat *etat* du
;; tableau lebensraum, en les affichant au fur et  mesure dans la fentre
;; *fen*.
;;   Un compteur de gnrations est affich dans la fentre de l'interprte.

(defun evolution (n)
  (let* ((win (front-window :class 'listener)) ;Listener le plus en avant plan :
         (buf (fred-buffer win))               ;ceci semble mieux que *top-listener*
         (pos (+ (buffer-position buf) 4))     ;Largeur 4 arbitraire, mais cohrente
         (nbr 0))
  (labels ((interne ()
             (affiche) (generation) ;;;(format t ".")
             (buffer-delete buf (- pos 4) pos) ;ces trois lignes : affichage
             (format t "~4D" (incf nbr))       ;no. de gnration -> listener
             (fred-update win)
             (dotimes (n *rall*) (* n 2))      ;07/2008 : ralentissement      
             (when (< nbr n) (interne))))
    (when (and (integerp n) (> n 0)) (format t "    ") (interne) t))))

;;   "Occupe" <n> cellules de l'tat <e> -- i.e. met leur valeur  1. Les
;; cellules sont dsignes alatoirement selon une distribution rectangulaire
;; uniforme. Il n'y a pas de vrification de la valeur prcdente des cellules,
;; donc pas forcment <n> cellules effectivement occupes au final.

(defun config-alea (n)
  (setq *etat* 0)
  (dotimes (i n 'ok)
    (setf (aref lebensraum *etat* (random COLONS) (random LIGNES)) 1)))

;;   Place une configuration de cellules occupes  une origine [<col>;<lin>]
;; donne de l'tat <e> du tableau lebensraum. L'origine/cellule [0;0] d'un tat
;; du tableau lebensraum correspondra au coin suprieur gauche de la fentre
;; d'affichage.
;;   La configuration proprement dite est dfinie par une liste <motif> de
;; coordonnes de cellules occupes reprsentes par des paires pointes :
;;    ((<c1> . <l1>) (<c2> . <l2>) ... (<cN> . <lN>)).
;; Ces coordonnes de points, constituant la dfinition d'une configuration,
;; sont relatives  son coin suprieur gauche.

(defun place (col lin motif)
  (setq *etat* 0)
  (labels ((interne ()
             (if (endp motif) 'ok
             (progn (setf (aref lebensraum *etat*
                                (mod (+ col (caar motif)) COLONS)
                                (mod (+ lin (cdr (pop motif))) LIGNES))
                          1)
                    (interne)))))
    (interne)
    'place))

;;   Trace une droite de cellules occupes (i.e. de valeur 1) entre les points
;; [<c1>;<l1>] et [<c2>;<l2>] de l'tat <e> du tableau lebensraum. Les deux
;; points donns sont inclus. La droite ne peut tre que horizontale, verticale,
;; ou diagonale  45.

(defun droite (c1 l1 c2 l2)
  (setq *etat* 0) (cond
  ((= c1 c2) (dotimes (n (1+ (- l2 l1)))
               (setf (aref lebensraum *etat* (mod c1 COLONS)
                                             (mod (+ l1 n) LIGNES))
                     1)))
  ((= l1 l2) (dotimes (n (1+ (- c2 c1)))
               (setf (aref lebensraum *etat* (mod (+ c1 n) COLONS)
                                             (mod l1 LIGNES))
                     1)))
  (t (dotimes (n (1+ (- c2 c1)))
               (setf (aref lebensraum *etat* (mod (+ c1 n) COLONS)
                                             (mod (+ l1 n) LIGNES))
                     1))))
'droite)

;;; ENVIRONNEMENT --------------------------------------------------------------
;;   Afin de pouvoir dplacer librement le directory cl.d sans avoir  diter
;; de paths dans le prsent fichier d'initialisation... (en cas de dmnagement,
;; installation sur autre disque ou ordinateur, etc.
(defvar common-lisp::*pathjeuvie*
  (merge-pathnames
   (make-pathname :directory
          (pathname-directory *load-truename*)) "*")) ; <--- PATH d'installation

;;   Lecture de configurations dans le dossier adjacent lifep provenant de
;; http://cafaq.com/lifefaq/index.php > lifep.zip. Le format de dfinition de
;; ces fichiers est dcrit dans lifep:WRITERS.txt. Cette collection de 174
;; configurations est due  Alan Hensel.
;;    Le "<fichier>".LIF sera lu et interprt de manire  placer la(les)
;; configuration(s) qu'il contient dans le tableau lebensraum.
;; Appel : (lifep <e> "<fichier>")
;;    => la(les) configuration(s) du "<fichier>" serons places dans l'tat <e>
;;       du tableau lebensraum, par rapport  une cellule centrale d'origine
;; Dans le cas o la taille d'une configuration dpasse celle du tableau
;; lebensraum, elle sera "replie et superpose" sur elle-mme, le tableau tant
;; trait comme une surface torique. 

(defun lifep (fichier &aux
                        listelue debutlue    ;pour lecture des lignes du fichier
                        origC origL          ;origine absolue dans lebensraum
                        origConfC origConfL  ;origine relative d'une config.
                        config               ;paires pointes la dcrivant
                        cmptlignes)          ;compteur de lignes de la config
  ;Origine absolue [0;0] au centre de lebensraum
  (setq origC (floor (/ COLONS 2)))
  (setq origL (floor (/ LIGNES 2)))
  (with-open-file
    (flux (cat-pathnames common-lisp::*pathjeuvie* "lifep"
                         (concatenate 'string fichier ".LIF"))
          :direction         :input
          :if-does-not-exist :error)
    (do ((ligne 1                        (1+ ligne))
         (lue (read-line flux nil 'okko) (read-line flux nil 'okko)))
        ((equal 'okko lue)
         ;Si config n'est pas nil, la configuration dj stocke prcdemment
         ;avant la fin du fichier doit tre place dans l'tat <e> du tableau
         ;lebensraum
         (when config (place (mod (+ origC origConfC) COLONS)
                             (mod (+ origL origConfL) LIGNES)
                             config))
         (format t "~&Fin normale fichier ~A, ~D lignes.~%" fichier (1- ligne)))
      (delete #\Linefeed lue)
      ;(format t "~&ligne~4D : ~17T~S~%" ligne lue)
      (setq listelue (separer-mots lue
            :separateurs '(#\space #\newline #\tab #\return #\linefeed #\page)))
      ;(format t "~17T~S~%" listelue)
      (setq debutlue (car listelue)) (cond

       ;Les lignes blanches sont ignores
       ((null listelue))

       ;Les lignes commenant par #Life (n de version), #D (description) ou #N
       ;(rgles normales "23/3" de J.H. Conway) sont simplement recopies 
       ;l'cran
       ((or (string= "#Life" debutlue)
            (string= "#D" debutlue) (string= "#N" debutlue))
        (format t "~A~%" lue))

       ;Une ligne commenant par #R (rgles) suivi d'autre chose que "23/3"
       ;indique une configuration conue pour d'autres rgles que les normales
       ;implmentes ici. Provoque donc arrt sur erreur. Les lignes "#N"
       ;(ci-dessus) ou "#R 23/3" sont quivalentes
       ((and (string= "#R" debutlue) (not (string= "23/3" (cadr listelue))))
        (format t "~A~%" lue)
        (error "Fichier concu pour autres regles que \"23/3\" de Conway : ~A."
               (cadr listelue)))

       ;Les lignes commenant par #P indiquent le dbut d'une configuration.
       ;#P est suivi des coordonnes <colonne> et <ligne> de la cellule o le
       ;coin suprieur gauche de la configuration (i.e. sa cellule (0 . 0)) doit
       ;tre plac dans l'espace lebensraum
       ((string= "#P" debutlue)
        ;Si config n'est pas nil, ce n'est pas la premire ligne commenant par
        ;#P, et la configuration dj stocke prcdemment doit tre place dans
        ;l'tat <e> du tableau lebensraum
        (when config (place (mod (+ origC origConfC) COLONS)
                            (mod (+ origL origConfL) LIGNES)
                            config))
        (setq origConfC (read-from-string (cadr listelue)))
        (setq origConfL (read-from-string (caddr listelue)))
        ;(dbg origConfC) (dbg origConfL)
        (setq config nil)
        (setq cmptlignes 0))

       ;Les lignes dfinissant une configuration, aprs une ligne #P, sont
       ;composes de caractres "." (cellules vides) et "*" (cellules occupes)
       ;juxtaposs. Les "*" sont traduites sous forme d'une liste de paires
       ;pointes (<colonne> . <ligne>). La configuration ainsi transcrite pourra
       ;tre ensuite positionne dans le tableau lebensraum par la fonction
       ;place dfinie ci-dessus
       ((or (find #\. debutlue) (find #\* debutlue))
        ;(format t "ici-->>>~A~%" lue)
        ;(dbg config)
        (dotimes (n (length debutlue) (incf cmptlignes))
          (when (char= #\* (elt debutlue n)) (push (cons n cmptlignes) config)))
        )
       (t (error "Ligne bizarre fichier ~A : ~S" fichier lue))
       )
      )
    )
)

;;   Utilitaire pour la prcdente.  partir d'une chane de caractres, donne
;; une liste de sous-chanes, ou <mots>. Un <mot> est compos d'un ou plusieurs
;; caractres non-sparateurs, spars par un ou plusieurs caractres
;; sparateurs. Le mot-clef :separateurs permet de modifier la liste des
;; caractres dfinis comme sparateurs.
;; Ex. : (separer-mots "    ")                   ==> nil
;;       (separer-mots " C'est lui ! Non ? ")    ==> ("C" "est" "lui" "Non")
;; Fait pour "CRIM" (Christian Morales-Ossio), 20/12/97.
(defun separer-mots (cadena
     &key (separateurs
               '(#\space #\newline #\tab #\return #\linefeed #\page ;blancs
                 #\, #\. #\; #\: #\! #\?                            ;ponctuations
                 #\- #\' #\( #\) #\" #\/                            ;grammaticaux
                ))  )
  (do ( (rez    ()                    )
        (mot    ""                    )
        (caract nil                   )
        (long   (length cadena)       )
        (n      0               (1+ n)) )
      ( (= n long) (if (string= "" mot) rez
                       (append rez (list mot))) )
    (if (member (setq caract (char cadena n)) separateurs)
      (when (not (string= "" mot))
        (setq rez (append rez (list mot)))
        (setq mot ""))
      (setq mot (concatenate 'string mot (string caract))))))

;;; Fonctions graphiques

;;   Charger la bibliothque QuickDraw.

(defun cat-pathnames (depart &rest frags)
  (labels ((interne (frgs ldr)
            (if (endp (cdr frgs))
              (merge-pathnames (car frgs) ldr)
            (interne (cdr frgs)
                     (merge-pathnames 
                      (make-pathname
                       :directory
                       (namestring (merge-pathnames (car frgs) ldr)))
                      "*")))))
  (interne frags depart)))
(load (cat-pathnames (mac-default-directory) "Library" "QuickDraw"))
(format t "~&;Bibliothque QuickDraw charge...~%")

;;   Cration d'une fentre pour la reprsentation graphique.
;; Horizontalement : <cells> colonnes ; verticalement : <etats> lignes.
;;   <titre> sera le titre de la fentre.
(defun cree-fenetre (cells etats titre &aux fen)
  (setq cells (* points cells))
  (setq etats (* points etats))
  (setq fen
    (make-instance 'window ;cf. p 154 classe window
                   :view-size (make-point cells etats)
                   :window-title titre
                   :view-font '("Webdings" 5 :srcCopy (:color-index 211))
                   :color-p t
                   :window-type :document
                   :close-box-p t))
  (set-fore-color fen 3740336)
  (set-back-color fen 13631336)
  (zigonne fen)                                            ;pour viter la perte
  (fill-rect fen *light-gray-pattern* 0 0 cells etats)     ;griser la fentre
;Ceci inutile, juste pour voir fentre lors des premiers tests...
;  (dotimes (n (/ etats points) 'ok) ;etats
;    (move-to fen Ox (+ Oy (* n points)))
;    (dotimes (n (/ cells points) 'ok) ;cellules
;      (format fen (if (string-equal car "g") (setq car "c") (setq car "g")))))
;
  fen
)

;;   Mouvement inutiles  effectuer aprs la cration de la fentre, vitant la
;; perte des quelques premiers affichages. Empirique et inexplicable !!!
(defun zigonne (fenetre)
  (dotimes (n 100001 t)                   ;100001 = empirique & inexplicable !!!
    (move fenetre 100 100)))

;;   Affiche l'tat *etat* du tableau lebensraum dans la fentre *fen*.
;; Amlioration 07/2008 : seules les cellules changeant d'tat sont rafrachies.
;; On obtient donc une vitesse d'volution condidrablement plus rapide.

(defun affiche (&aux autre etat)
  (setq autre (- 1 *etat*))                              ;autre = tat prcdent
  (dotimes (l LIGNES)
    (dotimes (c COLONS)
      (when (not (eq (setq etat (aref lebensraum *etat* c l))
                     (aref lebensraum autre c l)))
        (move-to *fen* (+ Ox (* c points)) (+ Oy (* l points)))
        (format *fen* (if (zerop etat) Wc Wg))))))
;;   Ancienne version :
;(defun affiche ()
;  (dotimes (l LIGNES)
;    (move-to *fen* Ox (+ Oy (* l points)))
;    (dotimes (c COLONS)
;        (format *fen* (if (zerop (aref lebensraum *etat* c l)) Wc Wg)))))

;;   07/2008 : fonction appele par vide-lebensraum. Ajoute pour permettre
;; amlioration de fonction affiche, ne rafrachissant plus que les cellules
;; changeant d'tat.

(defun affiche-vide ()
  (dotimes (l LIGNES)
    (move-to *fen* Ox (+ Oy (* l points)))
    (dotimes (c COLONS)
        (format *fen* Wc))))
;;   07/2008 : fonction de rglage de la boucle de ralentissement d'affichage,
;; dont la longueur est dans la variable globale *rall*. Ex. :
;;    (rall 1 6) => (setq *rall* 1000000)

(defun rall (b e)
  (setq *rall* (* b (expt 10 e))))

;;   Cration d'une fentre COLONS x LIGNES, et conservation de son pointeur
;; dans la variable globale *fen*.

(setq *fen* (cree-fenetre COLONS LIGNES "*Jeu de la vie* de J. H. Conway"))

(format t "Fentre *fen* : ~D colonnes x ~D lignes =>~45T~7D cellules et~%~
           ~45T~7D test par generation~%"
          COLONS LIGNES (* COLONS LIGNES) (* 8 COLONS LIGNES))