;;; Canons mettant en oeuvre diverses fonctions rcursives "chaotiques" et
;;; autres "attracteurs tranges"

;;   Gestion du module "canons-stochastiques", ncessaire pour la fonction
;; init-random (ainsi que frandom, si utilise).

(when (not (boundp 'common-lisp::*path-stochastique*))
      (setq common-lisp::*path-stochastique* (pathname ""))
      (format *error-output*
            "Attention : le path common-lisp::*path-stochastique* est nul :~
            ~%~12Terreur(s) fichier(s) possible(s)."))
(require "canons-stochastiques"
      (merge-pathnames "canons_base.lisp" common-lisp::*path-stochastique*))

;;; Germes standards

(let ((vvv (/ pi 4)))
  (defparameter *Nicolas2-state* vvv)
  (defparameter *ChpClwn-state* vvv)
  (defparameter *May-state* vvv)
  (defparameter *Metropolis-state* vvv)
  (defparameter *Henon-state*
    (make-array 2 :element-type 'float
                  :initial-contents (make-list 2 :initial-element vvv)))
  (defparameter *Lorenz-state*
    (make-array 3 :element-type 'float
                  :initial-contents (make-list 3 :initial-element vvv)))
  (defparameter *StandardMapping-state*
    (make-array 2 :element-type 'float
                  :initial-contents (make-list 2 :initial-element vvv))))

;;; Canons

;;   D'aprs Franois Nicolas (Doc. musicale, *Dans la distance*, IRCAM, 1993-
;; 1994).

(defun nicolas2 ()
"
;    nicolas2 => serie ex. simple etirement/repliement de F. Nicolas
                 par * 2 (mod 1), sur *Nicolas2-state*
"
  (setq *Nicolas2-state* (nth-value 1 (floor (* 2.0 *Nicolas2-state*)))))


(defun chpclwn ()
"
;    chpclwn => serie Chapeau de clown, sur *ChpClwn-state*
"
  (setq *ChpClwn-state* (if (< *ChpClwn-state* 1/2) (* 2 *ChpClwn-state*)
                        (* 2 (- 1 *ChpClwn-state*)))))


(defun may (r)
"
;    may <r> => serie population/logistique de May-Feigenbaum de coefficient
;               <r>, sur *May-state*
"
  (setq *May-state* (* r *May-state* (- 1.0 *May-state*))))



(defun metropolis (r)
"
;    metropolis <r> => serie de Metropolis, Stein & Stein de coefficient <r>,
;                      sur *Metropolis-state*
"
  (setq *Metropolis-state* (* r (sin (* pi *Metropolis-state*)))))

(defun henon (a b)
"
;    henon <a> <b> => serie de Henon (attracteur etrange) d'un vecteur de deux
;        nombres reels, sur *Henon-state*. Valeurs usuelles des coefficients :
;        <a> = 1.4, <b> = 0.3
"
  (let ((x (aref *Henon-state* 0))
        (y (aref *Henon-state* 1)) )
  (setq *Henon-state*
        (make-array 2 :element-type 'float
                      :initial-contents (list
                                          (+ y 1.0 (* (- a) x x))
                                          (* b x))))))

;;   Fonction de test des bornes de sries de Henon.
;;    tsthen <n> => donne les minima et maxima atteints par les deux valeurs des
;;                  vecteurs aprs <n> termes de la srie

(defun tsthen (nbrfx &aux
                (depart *Henon-state*)
                (prem (make-array 2
                        :initial-contents '(999999999.0 -999999999.0)))
                (seco (make-array 2
                        :initial-contents '(999999999.0 -999999999.0))) )
  (dotimes  (i nbrfx (prins depart "
                          " prem "
                          " seco))
            (henon 1.4 0.3)
            (when (< (aref *Henon-state* 0) (aref prem 0))
              (setf (aref prem 0) (aref *Henon-state* 0)))
            (when (> (aref *Henon-state* 0) (aref prem 1))
              (setf (aref prem 1) (aref *Henon-state* 0)))
            (when (< (aref *Henon-state* 1) (aref seco 0))
              (setf (aref seco 0) (aref *Henon-state* 1)))
            (when (> (aref *Henon-state* 1) (aref seco 1))
              (setf (aref seco 1) (aref *Henon-state* 1)))))

(defun lorenz (s r b &optional (dt 0.01))
"
;    lorenz <s> <r> <b> &optional <dt> => serie de Lorenz (attracteur etrange)
;        d'un vecteur de trois nombres reels, sur *Lorenz-state*. Valeurs
;        usuelles des coefficients :
;            <s> = 10.0, <r> = 28.0, <b> = 8/3
;        <dt> est une constante d'integration (methode d'Euler), a maintenir
;        dans de petites valeurs : des variables de la serie divergent
;        rapidement vers de tres grandes valeurs si dt>0.024 environ,
;        produisant un Floating point overflow -- cf. [Bidlack,1992:40]
"
  (let* ((x  (aref *Lorenz-state* 0))
         (y  (aref *Lorenz-state* 1))
         (z  (aref *Lorenz-state* 2))
         (x_ (* s (- y x)))
         (y_ (- (* r x) y (* x z)))
         (z_ (- (* x y) (* b z))))
  (setq *Lorenz-state*
        (make-array 3 :element-type 'float
                      :initial-contents (list (+ x (* x_ dt))
                                              (+ y (* y_ dt))
                                              (+ z (* z_ dt)))))))

;;   Fonction de test des bornes de sries de Lorenz.
;;    tstlor <n> &optional <deltat> =>  donne les minima et maxima atteints par
;;        les trois valeurs des vecteurs aprs <n> termes de la srie

(defun tstlor (nbrfx s r b &optional (deltat 0.01) &aux
                (depart *Lorenz-state*)
                (prem (make-array 2
                        :initial-contents '(999999999.0 -999999999.0)))
                (seco (make-array 2
                        :initial-contents '(999999999.0 -999999999.0)))
                (troi (make-array 2
                        :initial-contents '(999999999.0 -999999999.0))))
  (dotimes  (i nbrfx (prins depart "
                          " prem "
                          " seco "
                          " troi ))
            (lorenz s r b deltat)
            (when (< (aref *Lorenz-state* 0) (aref prem 0))
              (setf (aref prem 0) (aref *Lorenz-state* 0)))
            (when (> (aref *Lorenz-state* 0) (aref prem 1))
              (setf (aref prem 1) (aref *Lorenz-state* 0)))
            (when (< (aref *Lorenz-state* 1) (aref seco 0))
              (setf (aref seco 0) (aref *Lorenz-state* 1)))
            (when (> (aref *Lorenz-state* 1) (aref seco 1))
              (setf (aref seco 1) (aref *Lorenz-state* 1)))
            (when (< (aref *Lorenz-state* 2) (aref troi 0))
              (setf (aref troi 0) (aref *Lorenz-state* 2)))
            (when (> (aref *Lorenz-state* 2) (aref troi 1))
              (setf (aref troi 1) (aref *Lorenz-state* 2)))))

;;    Celui-ci pas implment dans exemples CLCE. <x> drive quasi constamment,
;; en croissant ou dcroissant, selon le signe de <mu>. <y> semble osciller en
;; positif et ngatif. Semble difficile  exploiter... D'aprs Francois Nicolas
;; (Doc. musicale, *Dans la distance*, IRCAM, 1993-1994).

(defun StandardMapping (mu)
"
;    StandardMapping <mu> => serie Standard Mapping d'un vecteur de deux nombres
;        reels, sur *StandardMapping-state*. Par ex. <mu> dans [-2pi,2pi]...
"
  (let* ((xv (aref *StandardMapping-state* 0))
         (yn (- (aref *StandardMapping-state* 1)
                (* (/ mu (* 2 pi)) (sin (* 2 pi xv)))))
         (xn (+ xv yn)) )
  (setf (aref *StandardMapping-state* 0) xn)
  (setf (aref *StandardMapping-state* 1) yn))
  *StandardMapping-state*)

;;   Fonction de test des bornes de sries Standard Mapping.
;;    tstSmap <n> => donne les minima et maxima atteints par les deux valeurs
;;                   des vecteurs aprs <n> termes de la srie

(defun tstSmap (mu nbrfx &aux
                (depart *StandardMapping-state*)
                (prem (make-array 2
                        :initial-contents '(999999999.0 -999999999.0)))
                (seco (make-array 2
                        :initial-contents '(999999999.0 -999999999.0))) )
  (dotimes  (i nbrfx (prins depart "
                          " prem "
                          " seco))
            (StandardMapping mu)
            (when (< (aref *StandardMapping-state* 0) (aref prem 0))
              (setf (aref prem 0) (aref *StandardMapping-state* 0)))
            (when (> (aref *StandardMapping-state* 0) (aref prem 1))
              (setf (aref prem 1) (aref *StandardMapping-state* 0)))
            (when (< (aref *StandardMapping-state* 1) (aref seco 0))
              (setf (aref seco 0) (aref *StandardMapping-state* 1)))
            (when (> (aref *StandardMapping-state* 1) (aref seco 1))
              (setf (aref seco 1) (aref *StandardMapping-state* 1)))))

;;; Initialisation et contrle des germes des diverses fonctions chaotiques

;;   Fonction d'initialisation.
;;    init-chaos <nom> &optional <n> : cf. doc. string ci-dessous

(defun init-chaos (nom &optional (n nil) &aux variable gstand galeat)
"
;   Initialisation (interactive ou non) des germes des fonctions chaotiques.
; Dans tous les cas, retourne le nouveau germe defini.
;    init-chaos <nm> &optional <n> => initialisation de *<nm>-state* selon <n> :
;        <n> absent ou nil => sequence interactive
;        <n> = 0           => initialisation standard : retour au germe initial
;        <n> nombre < 0    => initialisation aleatoire (selon appel a random)
;        si <n> est un nombre reel positif, ou autre objet approprie selon la
;        fonction concernee, il est utilise directement comme germe
"
  (cond ((eq nom 'nicolas2)
           (setq variable '*Nicolas2-state*)
           (setq gstand   (/ pi 4))
           (setq galeat   (random 1.0)))
        ((eq nom 'chpclwn)
           (setq variable '*ChpClwn-state*)
           (setq gstand   (/ pi 4))
           (setq galeat   (random 1.0)))
        ((eq nom 'may)
           (setq variable '*May-state*)
           (setq gstand   (/ pi 4))
           (setq galeat   (random 1.0)))
        ((eq nom 'metropolis)
           (setq variable '*Metropolis-state*)
           (setq gstand   (/ pi 4))
           (setq galeat   (random 1.0)))
        ((eq nom 'henon)
           (setq variable '*Henon-state*)
           (setq gstand   (make-array 2
                            :element-type 'float
                            :initial-contents
                              (make-list 2 :initial-element (/ pi 4))))
           (setq galeat   (make-array 2
                            :element-type 'float
                            :initial-contents
                              (make-list 2 :initial-element (random 1.0)))))
        ((eq nom 'lorenz)
           (setq variable '*Lorenz-state*)
           (setq gstand   (make-array 3
                            :element-type 'float
                            :initial-contents
                              (make-list 3 :initial-element (/ pi 4))))
           (setq galeat   (make-array 3
                            :element-type 'float
                            :initial-contents
                              (make-list 3 :initial-element (random 1.0)))))
        ((eq nom 'StandardMapping)
           (setq variable '*StandardMapping-state*)
           (setq gstand   (make-array 2
                            :element-type 'float
                            :initial-contents
                              (make-list 2 :initial-element (/ pi 4))))
           (setq galeat   (make-array 2
                            :element-type 'float
                            :initial-contents
                              (make-list 2 :initial-element (random 1.0)))))
        (t (error "init-chaos : ~A : fonction inconnue." nom)) )
  (cond ((and (numberp n) (zerop  n)) (set variable gstand))
        ((and (numberp n) (minusp n)) (set variable galeat))
        ((and (numberp n) (plusp  n) (not (or (eq nom 'henon)
                                              (eq nom 'lorenz)
                                              (eq nom 'StandardMapping))))
              (set variable n))
        ((and (arrayp n) (= 1 (array-rank n)) (= 2 (array-total-size n))
              (eq nom 'henon))
                                      (set variable n))
        ((and (arrayp n) (= 1 (array-rank n)) (= 3 (array-total-size n))
              (eq nom 'lorenz))
                                      (set variable n))
        ((and (arrayp n) (= 1 (array-rank n)) (= 2 (array-total-size n))
              (eq nom 'StandardMapping))
                                      (set variable n))
        ((null n)
                  (format *query-io*
                  "~%Type d'initialisation du germe ~A de la fonction chaotique ~A:~%~
                  0~30T=> retour au germe initial (~D)~%~
                  nombre < 0~30T=> aleatoire (selon appel a random)~%~
                  nombre reel > 0, ou~30T=> initialisation directe a ce germe~%~
                  array adequat, si approprie~%~
                  init-chaos ? " variable nom gstand)
                  (init-chaos nom (eval (read))))
         (t
                  (cerror "continuer en donnant un nouvel argument d'appel ~
                           a init-chaos. Pour plus de details, faire ~
                           (documentation 'init-chaos 'function)"
                           "Mauvais argument d'appel : ~A."
                           n)
                  (init-chaos))))

;;; Gestion du module "canons-chaos"

(provide "canons-chaos")

;;   "Documentation".

(setq Doc.Stoch (concatenate 'string Doc.Stoch
                                     (format nil ";; canons_chaos
;    nicolas2                  => F. Nicolas par * 2
;         *Nicolas2-state*     := ~S
;    chpclwn                   => Chapeau de clown
;         *ChpClwn-state*      := ~S
;    may <r>                   => population/logistique May-Feigenbaum coef. <r>
;        *May-state*           := ~S
;    metropolis <r>            => Metropolis, Stein & Stein coef. <r>
;        *Metropolis-state*    := ~S
;    henon <a> <b>             => vect. 2 reels ; vals. usuelles coefs. : <a>=1.4 : <b>=0.3
;        *Henon-state*         := ~S
;    lorenz <s> <r> <b> [<dt>] => vect. 3 reels ; vals. usuelles coefs. :
;                                 <s>=10.0 ; <r>=28.0 ; <b>=8/3.
;                                 <dt> := petite const. integ. => diverge trop si dt>0.024
;        *Lorenz-state*        := ~S
;    StandardMapping <mu>      => vect. 2 reels ; ex. <mu> dans [-2pi,2pi]...
;        *StandardMapping-state*         := ~S
;    init-chaos <nm> [(n nil)] := nil=>interact. ; 0=>initial ; <0=> alea. ; autre=>*<nm>-state*
" *Nicolas2-state* *ChpClwn-state* *May-state* *Metropolis-state*
  *Henon-state* *Lorenz-state* *StandardMapping-state*)))

;(format t "~A" Doc.Stoch)
